<?php
	/**
	 * Base JSON API client for Loorex.com.<br>
	 *
	 * @author   	loorex.com <support@loorex.com>
	 * @category 	API
	 * @version		1.1
	 * @since		12.01.2026
	 * @package		LoorexApi
	 * @link		http://loorex.com/docs/api.html
	 */

	define("_LOOREX_FILE_SIZE_LIMIT_IMG", 		1.5*1024*1024);
	define("_LOOREX_FILE_SIZE_LIMIT_MEDIA", 	3*1024*1024);

	abstract class LoorexApiBase {
		protected $apiUrl;
		protected $apiName;
		protected $accessToken;
		protected $idCounter;
		protected $errors;
		protected $warnings;
		protected $lastResponse;
		protected $lastResult;
		protected $mbString;
		protected $UploadImageExt = [
			'jpg', 'jpeg', 'png', 'gif',
		];
		protected $UploadMediaExt = [
			'mp3',
		];

		function __construct($accessToken) {
			$this->apiUrl		= 'https://loorex.com/api/json';
			$this->accessToken	= $accessToken;
			$this->apiName		= $this->getApiName();
			$this->idCounter	= 0;
			$this->lastResponse	= '';
			$this->mbString		= ((extension_loaded('mbstring')) ? true : false);
		}

		public function parmApiUrl($url = null) {
			if (!is_null($url)) {
				$this->apiUrl = $url;
			}
			return $this->apiUrl;
		}

		public function getIsErrors() {
			return ((count($this->errors)) ? true : false);
		}

		public function getErrors() {
			return $this->errors;
		}

		public function getIsWarnings() {
			return ((count($this->warnings)) ? true : false);
		}

		public function getWarnings() {
			return $this->warnings;
		}

		public function getRawResponse($json = false) {
			return (($json) ? json_decode($this->lastResponse, true) : $this->lastResponse);
		}

		public function getExtendedResponse() {
			return $this->lastResult;
		}

		public function loadMediaFile(&$parms, $relativeMediaUrl, $pathToFile) {
			$ret		= false;
			$sizeLimit	= $this->getSizeLimit($pathToFile);
			if ($sizeLimit && $relativeMediaUrl && $pathToFile) {
				$fileCnt	= file_get_contents($pathToFile);
				$ret		= $this->loadMediaContent($parms, $relativeMediaUrl, $fileCnt);
			}
			return $ret;
		}

		public function loadMediaContent(&$parms, $relativeMediaUrl, $fileContent) {
			if (!is_array($parms['resources'])) {
				$parms['resources'] = [];
			}
			$ret		= false;
			$sizeLimit	= $this->getSizeLimit($relativeMediaUrl);
			$urlParts	= parse_url($relativeMediaUrl);
			if ($this->mbString) {
				$cntSize = mb_strlen($fileContent, '8bit');
			} else {
				$cntSize = strlen($fileContent);
			}
			if (!$urlParts['scheme'] && $sizeLimit && $relativeMediaUrl && $cntSize && $cntSize <= $sizeLimit) {
				$parms['resources'][$relativeMediaUrl] = base64_encode($fileContent);
				$ret = true;
			}
			return $ret;
		}

		protected function getSizeLimit($filePath) {
			$ret		= 0;
			$pathInfo	= pathinfo($filePath);
			$ext		= strtolower($pathInfo['extension']);
			if (in_array($ext, $this->UploadImageExt)) {
				$ret = _LOOREX_FILE_SIZE_LIMIT_IMG;
			} else if (in_array($ext, $this->UploadMediaExt)) {
				$ret = _LOOREX_FILE_SIZE_LIMIT_MEDIA;
			}
			return $ret;
		}

		protected function sendMessage($method, $params) {
			$message = array(
				'jsonrpc'		=> '2.0',
				'accessToken'	=> $this->accessToken,
				'method'		=> $this->apiName.'.'.$method,
				'params'		=> $params,
				'id'			=> $this->genMsgId(),
			);

			$jsonStr = json_encode($message);

			$ret				= [];
			$this->lastResponse	= '';
			$this->lastResult	= false;
			$this->errors		= [];
			$this->warnings		= [];
			$page				= $this->getPageContent($jsonStr);
			if ($page['ret']) {
				$this->lastResponse	= $page['content'];
				$json				= json_decode($page['content'], true);
				$ret				= $json;
			} else {
				if ($page['error']['number']) {
					$ret['errors'][] = sprintf('CURL(%s) %s', $page['error']['number'], $page['error']['description']);
				} else {
					$ret['errors'][] = sprintf('Wrong HTTP answer code (%s)', $page['code']);
				}
			}
			if (isset($ret['errors']) && is_array($ret['errors'])) {
				$this->errors = $ret['errors'];
			}
			if (isset($ret['warnings']) && is_array($ret['warnings'])) {
				$this->warnings = $ret['warnings'];
			}
			if (isset($ret['result']) && !$this->getIsErrors()) {
				$ret = $ret['result'];
			} else {
				$ret = false;
			}
			$this->lastResult = $ret;
			return $ret;
		}

		protected function genMsgId() {
			$this->idCounter++;
			return $this->idCounter;
		}

		protected function getPageContent($postData) {

			$ch = curl_init();
			curl_setopt($ch, CURLOPT_HEADER, true);
			curl_setopt($ch, CURLOPT_URL, $this->apiUrl);

			curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);

			ob_start();
			$CurlRet		= curl_exec($ch);
			$content		= ob_get_clean();

			$headers		= '';
			$header_size	= curl_getinfo($ch, CURLINFO_HEADER_SIZE);
			$contentType 	= curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
			if ($header_size) {
				$headers	= substr($content, 0, $header_size);
				$content	= substr($content, $header_size);
			}
			$httpCode		= curl_getinfo($ch, CURLINFO_HTTP_CODE);

			$CurlError		= curl_error($ch);
			$CurlErrorNo	= curl_errno($ch);

			//echo sprintf("ErrorNo: %s, ErrorText: %s<br>", $CurlErrorNo, $CurlError);
			//echo "HEADERS:<br><pre>".$headers."</pre>";
			
			if (PHP_MAJOR_VERSION < 8) {
				curl_close($ch);
			}

			$ret = [
				'content'	=> $content,
				'headers'	=> $headers,
				'code'		=> $httpCode,
				'error'		=> [
					'number'		=> $CurlErrorNo,
					'description'	=> $CurlError,
				],
				'ret'		=> (('200' == $httpCode && $content) ? true : false),
			];

			return $ret;
		}

		abstract protected function getApiName();
	}
?>